/*  Copyright 2025 Leuze electronic GmbH + Co. KG
 *
 *  Licensed under the Apache License, Version 2.0
 */

#ifndef LEUZE_ETH_COMM_IFC_H
#define LEUZE_ETH_COMM_IFC_H

#include <string>
#include <vector>

#include <boost/optional.hpp>

#include "leuze_msgs/msg/status_profile_msg_rod.hpp"
#include "leuze_rod_driver/common.h"

namespace leuze_rod_ros2_drivers
{
    /************************************************************************************************************
     * @class EthCommIfc
     * 
     * @brief Common interface for a real ROD scanner and a simulation mode   
     ************************************************************************************************************/
    class EthCommIfc
    {
        public:

            /****************************************************************************************************/
        
            /**
             * @brief virtual constructor
             */
            virtual ~EthCommIfc() = default;

            /****************************************************************************************************/

            /**
             * @brief    connect to a ROD scanner and read its setting
             * @param    hostname
             * @param    port  
             * @return   true if success 
             */
            virtual bool connect(const std::string hostname, int port) = 0;

            /**
             * @brief    close used TCP socket
             * @return   true if success 
             */
            virtual bool closeSocketTCP() = 0;

            /**
             * @brief    close used UDP socket
             * @return   true if success 
             */
            virtual bool closeSocketUDP() = 0;

            /****************************************************************************************************/

            /**
             * @brief    get ROD scanner status
             * @return   return ROD scanner status 
             */
            virtual leuze_msgs::msg::StatusProfileMsgRod getStatusData() = 0;

            /**
             * @brief    load complete ROD scanner status data
             */
            virtual int readStatusData() = 0;

            /**
             * @brief    refresh ROD scanner status data
             */
            virtual int refreshStatusData() = 0;

            /**
             * @brief   send a TCP command
             * @param   cmd_id
             * @return  0 if success
             */
            virtual int sendTcpCmd(std::uint32_t cmd_id) = 0;

            /****************************************************************************************************/

            /**
             * @brief    start capturing of TCP measurement packets
             * @return   true if success
             */
            virtual bool startCapturingTCP() = 0;

            /**
             * @brief    start capturing of UDP measurement packets
             * @return   true if success
             */
            virtual bool startCapturingUDP() = 0;

            /**
             * @brief    stop capturing of packets
             * @return   state of capturing
             */
            virtual bool stopCapturing() = 0;

            /**
             * @brief    get packet capturing state
             * @return   state of capturing
             */
            virtual bool getCapturingState() = 0;

            /****************************************************************************************************/
            
            /**
             * @brief    get scan data
             * @return   scan data 
             */
            virtual ScanData getScan() = 0;

            /**
             * @brief    get data of full scan
             * @return   full scan data 
             */
            virtual ScanData getCompleteScan() = 0;

            /**
             * @brief    get number of available scans
             * @return   number of available scans 
             */
            virtual std::size_t getScansAvailable() const  = 0;

            /**
             * @brief    get number of available full scans
             * @return   number of available full scans 
             */
            virtual std::size_t getCompleteScansAvailable() const  = 0;
    };

} // end namespace leuze_rod_ros2_drivers

#endif // LEUZE_eth_comm_rod_H
