# ROS 2 Driver for Leuze ROD x08 Laser Scanners

This repository contains all ROS 2 packages required for the ROD 308 and ROD 508 laser scanners. The driver is released under the [Apache license, Version 2.0](https://www.apache.org/licenses/LICENSE-2.0). Please read the file LICENSE before using or modifying the driver.

<div align="center">
    <img src="leuze_description/doc/rod_x08.jpg" alt="ROD 308 Laser scanner" width="400">
</div>

The ROD x08 driver consists of following packages:

*   `leuze_bringup`     - launch script for the driver
*   `leuze_description` - visualisation launch script and the 3D model
*   `leuze_msgs`        - custom message "StatusProfileMsgRod"
*   `leuze_rod_driver`  - source codes of the driver

## Scanner Setup

A ROD x08 laser scanner can be configured using the ROD Config Software. This software is available for download from the [Leuze ROD 308](https://www.leuze.com/en-int/rod-308/50153046) or [Leuze ROD 508](https://www.leuze.com/en-int/rod-508/50153047) web pages. These pages also contain additional information about the ROD x08 laser scanners.

<div align="center">
    <img src="leuze_description/doc/rod_config_sw_1.PNG" alt="ROD Config Software" >
</div>

Using the ROD Config Software, you can set the scanner's name and the following scan parameters. These are automatically communicated to the ROS 2 driver at startup, so there is no need to manually copy them into configuration files or source code.

*   `Scanning Angle Range`
*   `Scanning Frequency / Angular Resolution`
*   `Skip Spot`
*   `Filter Type` and its parameters
*   `Contamination threshold` for warning and error
*   `Data packet type`
*   `Data output direction`
*   `Status LED`

To set the ROD x08 laser scanner IP address and port, go to: Connection → Network Settings.

<div align="center">
    <img src="leuze_description/doc/rod_config_sw_2.PNG" alt="ROD Config Software - Network settings" >
</div>

You can configure:

*   `IP address`
*   `Subnet mask`
*   `Default gateway`
*   `Port`

Note: Only one port needs to be set. This port is used for both UDP and TCP communication. Refer to the "ROD-300-500 Communication Protocol" document available on the Leuze [Leuze ROD 308](https://www.leuze.com/en-int/rod-308/50153046) or [Leuze ROD 508](https://www.leuze.com/en-int/rod-508/50153047) web page for more details. The only settings required by the ROS 2 driver are the `IP address` and the `Port`.

## Driver Installation

Follow the steps below to install the ROS 2 driver for Leuze ROD x08 laser scanners.

1. Activate the ROS2 first:

    ```
    source /opt/ros/<distro>/setup.bash
    ```
    Replace <distro> with your ROS 2 distribution (e.g., humble). This command must be run in each new terminal session.

2. Create a ROS 2 workspace and navigate to the `src` folder:

    ```
    mkdir -p ~/ros2_ws/src
    cd ~/ros2_ws/src/
    ```

3. Clone the driver source code from GitHub:

    ```
    git clone https://github.com/thesensorpeople/leuze_rod_ros2_drivers.git
    ```

4. Install dependencies using `rosdep`:

    ```
    cd ..
    rosdep install --from-paths src --ignore-src -r -y
    ```

5. Build the driver and source the workspace:

    ```
    colcon build
    source install/setup.bash
    ```

    Alternatively, to build only specific packages:

    ```
    colcon build --packages-select leuze_bringup
    source install/setup.bash

    colcon build --packages-select leuze_description
    source install/setup.bash

    colcon build --packages-select leuze_msgs
    source install/setup.bash

    colcon build --packages-select leuze_rod_driver
    source install/setup.bash
    ```

### Driver Packages

The ROS 2 driver consists of the following packages:

- `leuze_bringup`: Launch file for starting the ROS 2 driver for ROD x08 laser scanners. 
- `leuze_description` : 3D models of ROD x08 laser scanners and a launch file for RViz visualization. 
- `leuze_msgs` : Custom message definition StatusProfileMsgRod for publishing configuration and status data.
- `leuze_rod_driver` : Core driver source code for Leuze ROD x08 laser scanners.

## Driver Modes

There are three optional driver modes that can be enabled by editing `leuze_rod_driver/CMakeLists.txt`.

### 1. Debugging Mode

Enabling this mode adds additional log output to the terminal. It is not recommended for use in production. To enable, uncomment the following line:

```
# add_definitions(-DDEBUG)                            # enable debug logs
```

Then rebuild and source the workspace:

```
colcon build
source install/setup.bash
```

### 2. Simulation Mode

Useful for testing node setups and verifying communication between components without hardware. To enable, uncomment:

```
# add_definitions(-DSIMULATION)                       # enable simulation mode
```

Then rebuild and source:

```
colcon build
source install/setup.bash
```

<div align="center">
    <img src="leuze_description/doc/driver_simulation.PNG" alt="ROS2 driver for ROD x08 Laser scanner - Simulation" >
</div>

### 3. Status publishing Mode

Enabling this mode activates cyclic communication with the TCP command interface of a ROD x08 laser scanner. In this mode, the driver cyclically reads status data from the scanner and publishes it to a topic named `..._status`, where `...` corresponds to the name of the topic used for publishing scan data. To enable, uncomment:

```
# add_definitions(-DSTATUS)                          # enable publishing of status messages
```

Then rebuild and source:

```
colcon build
source install/setup.bash
```

Check `leuze_msgs/msg/StatusProfileMsgRod.msg` for information about published data items and structure.

## Driver Execution

It is possible to execute the driver using a launch script, which is a part of the ROS2 package leuze_bringup.

```
ros2 launch leuze_bringup leuze_bringup_rod.launch.py 
```

This launches the driver with the following default settings:

*   IP Address: `192.168.60.101`
*   Port:       `3050`
*   Topic ID:   `scan1`
*   Frame ID:   `scanner_laser`

### Parameters

If the default settings don't apply, only the following parameters must be specified. The driver will automatically retrieve all other settings from the scanner:

*   `ip_addr`:  Scanner IP address
*   `port_num`: Port number
*   `topic_id`: ROS 2 topic name to publish scan data
*   `frame_id`: Frame ID associated with the scan data

Example:
```
ros2 launch leuze_bringup leuze_bringup_rod.launch.py ip_addr:=192.168.2.38 port_num:=1234 topic_id:=scan1 frame_id:=scanner_laser
```

You can run multiple scanner instances simultaneously by assigning unique topic and frame IDs.

<div align="center">
    <img src="leuze_description/doc/driver_real_scanner.PNG" alt="ROS2 driver for ROD x08 Laser scanner" >
</div>

### Running the Driver Without a Launch File

The driver can also be run directly:

```
colcon build
source install/setup.bash
ros2 run leuze_rod_driver leuze_rod_driver
```
To run it with specific parameters:

```
ros2 run leuze_rod_driver leuze_rod_driver <IP_address> <port_num> <topic_id> <frame_id>
```
Example:

```
ros2 run leuze_rod_driver leuze_rod_driver 192.168.2.38 3050 scan1 scanner_laser
```

## Visualization

To launch a visualization using RViz 2 and the scanner model:

```
 ros2 launch leuze_description view_rod300_500.launch.py
```
This opens RViz 2 with a preloaded ROD x08 scanner model. Two fixed frames are available:

*   `scanner_laser`         - laser beams positioned at height 0
*   `scanner_mount_link`    - laser beams positioned at a realistic mounting height
                            (e.g., floor-mounted).

### Optional Parameters

*   `use_rviz`      - Launch RViz automatically (default: `True`)
*   `use_sim_time`  - Use simulated time instead of real-time (default: `False`)

To see the scan data in the RViz2 window it as necessary to execute also the driver for the ROD x08 Laser scanner in a separated terminal.

<div align="center">
    <img src="leuze_description/doc/vizualisation.PNG" alt="ROS2 driver for ROD x08 Laser scanner - Visualization" >
</div>

---

© 2025 **Leuze electronic GmbH + Co. KG**.     
Visit [www.leuze.com](https://www.leuze.com) | Contact: [info@leuze.com](mailto:info@leuze.com)  